#include <stdlib.h>
#include <getopt.h>

// some utility functions

static int mem_equal(const void *m1, const void *m2, size_t len)
{
	return memcmp(m1, m2, len) == 0;
}

static int string_equal(const char *s1, const char *s2)
{
	return strcmp(s1, s2) == 0;
}

template <class T>
static void mem_zero(T (&buff))
{
	memset(&buff, 0, sizeof(T));
}

static long file_size(FILE *f)
{
	long savedPos = ftell(f);
	fseek(f, 0, SEEK_END );
	long size = ftell(f);
	fseek(f, savedPos, SEEK_SET );
	return(size);
}

#define printfn(msg, ...) printf(msg "\n", ## __VA_ARGS__)

#ifndef NDEBUG
	#define debug_printfn(msg, ...) printf(msg "\n", ## __VA_ARGS__)
#else
	#define debug_printfn(msg, ...) { }
#endif

#define iterateTimes(limit, iterator) for(typeof (limit) iterator = 0; iterator < (limit); iterator++)

// common bluetooth functions/data

static const char *aboutStr = "iControlPad firmware updater 1.1 by Robert Broglia";
static const char *fwPath = "icp_firmware.enc";

static bool btAddrIsSet(const char *str)
{
	return !string_equal(str, "XX:XX:XX:XX:XX:XX");
}

static char btAddrStr[] = "XX:XX:XX:XX:XX:XX";
static const char *btName =
#ifdef DEFAULT_ICONTROLPAD
	"iControlPad-"
#else
	0
#endif
;
static bool btNameIsSubstring =
#ifdef DEFAULT_ICONTROLPAD
	1
#else
	0
#endif
;

static const unsigned char noCodeClass[] = { 0x00, 0x1F, 0x00 };
static const unsigned int noCodeClassInt = 0x1F00;
static const unsigned char flashOkReply = 0x11;

static void processBTAddrStr(const char *addr);

static bool deviceNameMatches(const char *name)
{
	return (btNameIsSubstring && strstr(name, btName)) ||
			(!btNameIsSubstring && string_equal(name, btName));
}

static void parseArgs(int argc, char *argv[])
{
	int c;
	while ((c = getopt (argc, argv, "a:n:")) != -1)
	switch (c)
	{
		case 'a':
			processBTAddrStr(optarg);
		break;
		
		case 'n':
			btName = optarg;
			btNameIsSubstring = 0;
		break;
		
		case '?':
			if (optopt == 'a')
				printfn("Please specify an address for -a in the form XX:XX:XX:XX:XX:XX");
			else if (optopt == 'n')
				printfn("Please specify a name for -n");
			else if (isprint (optopt))
				printfn("Unknown option `-%c'", optopt);
			else
				printfn("Unknown option character `\\x%x'", optopt);
		exit(1);
			
		default:
		abort();
	}
 
	for(int index = optind; index < argc; index++)
	{
		fwPath = argv[index];
		debug_printfn("set firmware path: %s", fwPath);
		break;
	}
	
	if(!btAddrIsSet(btAddrStr) && !btName)
	{
		printfn("Error, you must specify at least an address (-a) or device name (-n)");
		exit(1);
	}
}

static bool processFirmwareFile(FILE *fw, unsigned char *fwData, unsigned int fwSize)
{	
	unsigned int tempVal, readFwSize = 0;
	while(fscanf(fw, "%2X", &tempVal) == 1)
	{
		if(readFwSize == fwSize)
		{
			printfn("Error, firmware overflows %d byte buffer", fwSize);
			return 0;
		}
		fwData[readFwSize] = tempVal;
		readFwSize++;
	}
	if(fwSize != readFwSize)
	{
		printfn("Expected %d byte firmware, but read %d bytes", fwSize, readFwSize);
		return 0;
	}

	printfn("Loaded %d byte firmware...", fwSize);
	fclose(fw);
	return 1;
}

