/* iControlPad Firmware updater 1.0.0 by Robert Broglia */

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/rfcomm.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
#include <ctype.h>

#include "common.hh"

static void fd_setNonblock(int fd, bool on)
{
	int flags;
	flags = fcntl(fd,F_GETFL,0);
	assert(flags != -1);
	if(on)
		fcntl(fd, F_SETFL, flags | O_NONBLOCK);
	else
		fcntl(fd, F_SETFL, flags & ~O_NONBLOCK);
}

static bdaddr_t btAddr;
static void processBTAddrStr(const char *addr)
{
	if(str2ba(addr, &btAddr) != 0)
	{
		printfn("Invalid address specified, use the form XX:XX:XX:XX:XX:XX");
		exit(1);
	}
	strncpy(btAddrStr, addr, sizeof(btAddrStr));
	btAddrStr[sizeof(btAddrStr)-1] = 0;
}

static bool findBTDev(int hciDevId, int hciSocket, const char *name, bdaddr_t &bdaddrOut)
{
	printfn("Starting Bluetooth scan...");
	int devices = 0, maxDevices = 25;
	inquiry_info *deviceInfo = 0;
	devices = hci_inquiry(hciDevId, 4, maxDevices, 0, &deviceInfo, IREQ_CACHE_FLUSH);
	if(devices == -1)
	{
		printfn("Error, scan failed");
		return 0;
	}
	if(devices == 0)
	{
		printfn("No devices present");
		return 0;
	}

	printfn("Found %d device(s)", devices);
	iterateTimes(devices, i)
	{
		#ifdef DEFAULT_ICONTROLPAD
		if(!mem_equal(deviceInfo[i].dev_class, noCodeClass, 3))
		{
			printfn("Skipping device %d due to class %X:%X:%X", i, deviceInfo[i].dev_class[0], deviceInfo[i].dev_class[1], deviceInfo[i].dev_class[2]);
			continue;
		}
		#endif
		char name[248];
		if(hci_read_remote_name(hciSocket, &deviceInfo[i].bdaddr, sizeof(name), name, 0) < 0)
			strcpy(name, "Unknown");
		printfn("#%d name: %s", i, name);

		if(deviceNameMatches(name))
		{
			printfn("Using this device...");
			bacpy(&bdaddrOut, &deviceInfo[i].bdaddr);
			return 1;
		}
	}
	// TODO: cleanup deviceInfo
	return 0;
}



int main(int argc, char *argv[])
{
	printfn("%s", aboutStr);
	parseArgs(argc, argv);
	
	FILE *fw = fopen(fwPath, "rb");
	if(!fw)
	{
		printfn("Error opening firmware file: %s", fwPath);
		return 1;
	}
	
	unsigned int fwSize = file_size(fw)/2; // should be half to file size after reading hex digit pairs
	if(!fwSize)
	{
		printfn("Error, firmware file too small: %s", fwPath);
		return 1;
	}
	
	unsigned char fwData[fwSize];
	
	if(!processFirmwareFile(fw, fwData, fwSize))
		return 1;

	// open BT
	int hciDevId = hci_get_route(0);
	if(hciDevId < 0)
	{
		printfn("Error opening bluetooth hardware");
		return 1;
	}
	int hciSocket = hci_open_dev(hciDevId);
	if(hciSocket < 0)
	{
		printfn("Error opening socket");
		return 1;
	}
	
	// Scan for device if address not provided
	if(!btAddrIsSet(btAddrStr))
	{
		assert(btName);
		if(!findBTDev(hciDevId, hciSocket, btName, btAddr))
		{
			printfn("Error, device name %s not found", btName);
			return 1;
		}
		ba2str(&btAddr, btAddrStr);
	}
	
	printfn("Connecting to %s...", btAddrStr);
	int sock;
	struct sockaddr_rc addr;
	mem_zero(addr);
	addr.rc_family = AF_BLUETOOTH;
	addr.rc_channel = (uint8_t)1;
	addr.rc_bdaddr = btAddr;
	sock = ::socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
	if(sock == -1)
	{
		printfn("Error creating socket");
		return 1;
	}
	if(connect(sock, (struct sockaddr *)&addr, sizeof addr) == -1)
	{
		printfn("Error connecting socket");
		return 1;
	}

	printfn("Uploading firmware...");
	
	unsigned int frameSize;
	unsigned char receivedByte;
	unsigned int frames = 0;
	for(unsigned int index = 0; index < fwSize; index += frameSize, frames++)
	{
		frameSize = ((fwData[index] << 8) | fwData[index + 1]) + 2;

		fd_setNonblock(sock, 1);
		while(read(sock, &receivedByte, 1) > 0)
			printfn("Ignoring extra byte 0x%X from device", (int)receivedByte);
		fd_setNonblock(sock, 0);
		
		printfn("Writing frame %d: 0x%X+0x%X, %d%%", frames, index, frameSize, 100 * (index+frameSize) / fwSize);
		int writeResult = write(sock, &fwData[index], frameSize);
		if(writeResult != (int)frameSize)
		{
			printfn("Error writing, write() returned %d", writeResult);
			return 1;
		}

		// Check response
		if(read(sock, &receivedByte, 1))
		{
			debug_printfn("read back %d", receivedByte);
			if(receivedByte == flashOkReply)
			{
				debug_printfn("response ok");
			}
			else
			{
				printfn("CRC error, please verify your firmware file and try again");
				return 1;
			}
		}
		else
		{
			printfn("Error reading back response from device");
			return 1;
		}
	}
	printfn("Finished successfully");
	return 0;
}

