#import <Foundation/NSObject.h>
#import <IOBluetooth/objc/IOBluetoothDevice.h>
#import <IOBluetooth/objc/IOBluetoothDeviceInquiry.h>
#import <IOBluetooth/objc/IOBluetoothRFCOMMChannel.h>
#import <IOBluetooth/IOBluetoothUtilities.h>
#import "common.hh"

static unsigned int fwSize;
static BluetoothDeviceAddress btAddr;
static unsigned char *fwDataPtr;

static void processBTAddrStr(const char *addr)
{
	NSString *addrStr = [NSString stringWithCString: addr encoding: NSUTF8StringEncoding];
	IOBluetoothNSStringToDeviceAddress(addrStr, &btAddr);
	strncpy(btAddrStr, addr, sizeof(btAddrStr));
	btAddrStr[sizeof(btAddrStr)-1] = 0;
}

static void writeFrame(IOBluetoothRFCOMMChannel* ch)
{
	static unsigned int frames = 0, index = 0;
	if(index >= fwSize)
	{
		printfn("Finished successfully");
		exit(0);
	}
	unsigned int frameSize = ((fwDataPtr[index] << 8) | fwDataPtr[index + 1]) + 2;
	printfn("Writing frame %d: 0x%X+0x%X, %d%%", frames, index, frameSize, 100 * (index+frameSize) / fwSize);
	[ch writeSync : &fwDataPtr[index] length: frameSize];
	index += frameSize; frames++;
}

@interface FirmwareFlasher : NSObject {}

-(void)rfcommChannelOpenComplete: (IOBluetoothRFCOMMChannel*)channel
status: (IOReturn)status ;

-(void)rfcommChannelData : (IOBluetoothRFCOMMChannel*)channel
data: (void*)dataPointer
length: (size_t)dataLength;

@end

@implementation FirmwareFlasher

-(void)rfcommChannelOpenComplete : (IOBluetoothRFCOMMChannel*)channel
status: (IOReturn)status
{
	if(kIOReturnSuccess == status)
	{
		printfn("Uploading firmware...");
		writeFrame(channel);
	}
	else
	{
		printfn("Connection error");
		exit(1);
	}
}

-(void)rfcommChannelData : (IOBluetoothRFCOMMChannel*)channel
data: (void*)dataPointer
length: (size_t)dataLength
{
	if(((char*)dataPointer)[0] != flashOkReply)
	{
		printfn("CRC error, please verify your firmware file and try again");
		[channel closeChannel];
		exit(0);
	}
	else
	{
		writeFrame(channel);
	}
}

@end

static bool foundDevice = 0;

static void openConnection()
{
	printfn("Connecting to %s...", btAddrStr);
	IOBluetoothDevice *device = [IOBluetoothDevice withAddress: &btAddr];
	IOBluetoothRFCOMMChannel *chan;
	FirmwareFlasher *handler = [[FirmwareFlasher alloc] init];
	if([device openRFCOMMChannelSync: &chan withChannelID: 1 delegate: handler] != kIOReturnSuccess)
	{
		printfn("Error opening channel");
		exit(1);
	}
}

@interface BTScanner : NSObject {}

-(void) deviceInquiryComplete: (IOBluetoothDeviceInquiry*) sender
error: (IOReturn) error
aborted: (BOOL) aborted;

-(void) deviceInquiryDeviceFound: (IOBluetoothDeviceInquiry*) sender
device: (IOBluetoothDevice*) device;

@end


@implementation BTScanner

-(void) deviceInquiryComplete: (IOBluetoothDeviceInquiry*) sender
error: (IOReturn) error
aborted: (BOOL) aborted
{
	if(!foundDevice)
	{
		printfn("No usuable devices found");
		CFRunLoopStop(CFRunLoopGetCurrent()) ;
	}
}

-(void) deviceInquiryDeviceFound: (IOBluetoothDeviceInquiry * ) sender
device: (IOBluetoothDevice* ) device
{
	static int scannedDevs = 0;
	scannedDevs++;
	unsigned int cls = [device getClassOfDevice];
	#ifdef DEFAULT_ICONTROLPAD
	if(cls != noCodeClassInt)
	{
		printfn("#%d: skipping device due to class 0x%X", scannedDevs, cls);
		return;
	}
	#endif
	const char *addrStr = [[device getAddressString] cStringUsingEncoding: NSUTF8StringEncoding];
	printf("#%d: address: %s, name: ", scannedDevs, addrStr);
	[device remoteNameRequest: 0];
	const char *name = [device.name cStringUsingEncoding: NSUTF8StringEncoding];
	printfn("%s", name);
	
	if(deviceNameMatches(name))
	{
		foundDevice = 1;
		[sender stop];
		printfn("Using this device...");
		btAddr = *[device getAddress];
		strncpy(btAddrStr, addrStr, sizeof(btAddrStr));
		btAddrStr[sizeof(btAddrStr)-1] = 0;
		openConnection();
	}
}

@end

int main(int argc, char *argv[])
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	
	printfn("%s", aboutStr);
	parseArgs(argc, argv);
	
	FILE *fw = fopen(fwPath, "rb");
	if(!fw)
	{
		printfn("Error opening firmware file: %s", fwPath);
		return 1;
	}
	
	fwSize = file_size(fw)/2; // should be half to file size after reading hex digit pairs
	if(!fwSize)
	{
		printfn("Error, firmware file too small: %s", fwPath);
		return 1;
	}
	
	unsigned char fwData[fwSize];
	
	if(!processFirmwareFile(fw, fwData, fwSize))
		return 1;
	fwDataPtr = fwData;
	
	if(!btAddrIsSet(btAddrStr))
	{
		printfn("Starting Bluetooth scan...");
		BTScanner *d = [[BTScanner alloc] init];
		IOBluetoothDeviceInquiry *bdi = [[ IOBluetoothDeviceInquiry alloc] init];
		[bdi setDelegate : d];
		[bdi start];
	}
	else
	{
		openConnection();
	}
	CFRunLoopRun();
	[pool release];
	return 0;
}

